const express = require('express');
const mysql = require('mysql2/promise');
const http = require('http');
const socketIo = require('socket.io');
const cors = require('cors');
const path = require('path');

const app = express();
const server = http.createServer(app);
const io = socketIo(server, {
    cors: {
        origin: "*",
        methods: ["GET", "POST"]
    }
});

// Middleware
app.use(cors());
app.use(express.json());
app.use(express.static('public'));

// FORCE HTTPS REDIRECT MIDDLEWARE - CRITICAL FOR WEBRTC
app.use((req, res, next) => {
    // Check if request is over HTTP and not from localhost
    if (req.header('x-forwarded-proto') !== 'https' && 
        req.hostname !== 'localhost' && 
        req.hostname !== '127.0.0.1' && 
        process.env.NODE_ENV === 'production') {
        
        // Redirect to HTTPS
        return res.redirect(301, `https://${req.hostname}${req.url}`);
    }
    next();
});

// FIXED MySQL Database Configuration for "meeting" database
const dbConfig = {
    host: 'localhost',
    user: 'meeting',
    password: 'meeting', // Update this with your MySQL password if needed
    database: 'meeting' // Using your existing "meeting" database
};

// Database Connection Pool
const pool = mysql.createPool(dbConfig);

// Helper function to determine the correct protocol
function getSecureProtocol(req) {
    // Check various headers that might indicate HTTPS
    const forwardedProto = req.get('x-forwarded-proto');
    const forwardedSsl = req.get('x-forwarded-ssl');
    const cloudFrontProto = req.get('cloudfront-forwarded-proto');
    
    // If any indicator shows HTTPS, or if it's production, force HTTPS
    if (forwardedProto === 'https' || 
        forwardedSsl === 'on' || 
        cloudFrontProto === 'https' ||
        req.secure ||
        process.env.NODE_ENV === 'production') {
        return 'https';
    }
    
    // For localhost/development, allow HTTP
    if (req.hostname === 'localhost' || req.hostname === '127.0.0.1') {
        return req.protocol;
    }
    
    // Default to HTTPS for WebRTC compatibility
    return 'https';
}

// FIXED Database Initialization - No prepared statements for CREATE/USE
async function initializeDatabase() {
    try {
        console.log('🔄 Initializing database...');
        
        // Test connection to existing "meeting" database
        const testConnection = await pool.getConnection();
        console.log('✅ Database connection successful');
        console.log('✅ Using existing "meeting" database');
        
        // Create sessions table
        await testConnection.execute(`
            CREATE TABLE IF NOT EXISTS sessions (
                id VARCHAR(50) PRIMARY KEY,
                admin_username VARCHAR(100),
                status ENUM('active', 'waiting', 'ended') DEFAULT 'waiting',
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
            )
        `);
        console.log('✅ Sessions table created/verified');
        
        // Create customers table
        await testConnection.execute(`
            CREATE TABLE IF NOT EXISTS customers (
                id INT AUTO_INCREMENT PRIMARY KEY,
                session_id VARCHAR(50),
                name VARCHAR(100) NOT NULL,
                support_code VARCHAR(100),
                socket_id VARCHAR(100),
                status ENUM('connected', 'disconnected') DEFAULT 'connected',
                is_sharing_screen BOOLEAN DEFAULT FALSE,
                connected_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                last_seen TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                INDEX idx_session_id (session_id),
                INDEX idx_status (status)
            )
        `);
        console.log('✅ Customers table created/verified');
        
        // Create chat_messages table
        await testConnection.execute(`
            CREATE TABLE IF NOT EXISTS chat_messages (
                id INT AUTO_INCREMENT PRIMARY KEY,
                session_id VARCHAR(50),
                sender_type ENUM('admin', 'customer', 'system'),
                sender_name VARCHAR(100),
                message TEXT NOT NULL,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                INDEX idx_session_id (session_id)
            )
        `);
        console.log('✅ Chat messages table created/verified');
        
        // Create screen_sharing table
        await testConnection.execute(`
            CREATE TABLE IF NOT EXISTS screen_sharing (
                id INT AUTO_INCREMENT PRIMARY KEY,
                session_id VARCHAR(50),
                customer_id INT,
                status ENUM('active', 'stopped', 'requested') DEFAULT 'stopped',
                requested_by VARCHAR(100),
                started_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                INDEX idx_session_id (session_id)
            )
        `);
        console.log('✅ Screen sharing table created/verified');
        
        // Verify tables exist
        const [tables] = await testConnection.execute('SHOW TABLES');
        const tableNames = tables.map(t => Object.values(t)[0]);
        console.log('📊 Available tables:', tableNames);
        
        // Check specifically for our tables
        const requiredTables = ['sessions', 'customers', 'chat_messages', 'screen_sharing'];
        const existingTables = tableNames.filter(name => requiredTables.includes(name));
        console.log('✅ Remote support tables:', existingTables);
        
        testConnection.release();
        console.log('✅ Database initialization completed successfully');
        
    } catch (error) {
        console.error('❌ Database initialization error:', error.message);
        console.error('💡 Error details:', error);
        throw error; // Re-throw to handle in startServer
    }
}

// Test database connection
async function testDatabaseConnection() {
    try {
        const connection = await pool.getConnection();
        console.log('🔗 Database connection test: SUCCESS');
        
        // Test query
        const [rows] = await connection.execute('SELECT 1 as test');
        console.log('📋 Database query test: SUCCESS');
        
        connection.release();
        return true;
    } catch (error) {
        console.error('🔗 Database connection test: FAILED');
        console.error('Error:', error.message);
        return false;
    }
}

// API Routes with better error handling

// Admin login
app.post('/api/admin/login', async (req, res) => {
    const { username, password } = req.body;
    
    console.log(`🔐 Admin login attempt: ${username}`);
    
    if (username === 'admin' && password === 'admin123') {
        console.log('✅ Admin login successful');
        res.json({ success: true, username });
    } else {
        console.log('❌ Admin login failed - invalid credentials');
        res.status(401).json({ success: false, message: 'Invalid credentials' });
    }
});

// FIXED: Generate session with HTTPS URLs for WebRTC compatibility
app.post('/api/admin/generate-session', async (req, res) => {
    try {
        const { adminUsername } = req.body;
        const sessionId = 'SUP-' + Math.random().toString(36).substr(2, 9).toUpperCase();
        
        console.log(`🔄 Creating session ${sessionId} for admin ${adminUsername}`);
        
        const [result] = await pool.execute(
            'INSERT INTO sessions (id, admin_username, status) VALUES (?, ?, ?)',
            [sessionId, adminUsername, 'waiting']
        );
        
        console.log(`✅ Session ${sessionId} created successfully`);
        
        // CRITICAL FIX: Force HTTPS for share URL since WebRTC requires it
        const protocol = getSecureProtocol(req);
        const shareUrl = `${protocol}://${req.get('host')}?session=${sessionId}`;
        
        console.log(`🔒 Generated share URL: ${shareUrl} (Protocol: ${protocol})`);
        
        res.json({
            success: true,
            sessionId,
            shareUrl: shareUrl,
            protocol: protocol, // For debugging
            originalProtocol: req.protocol, // For debugging
            headers: {
                'x-forwarded-proto': req.get('x-forwarded-proto'),
                'x-forwarded-ssl': req.get('x-forwarded-ssl'),
                'host': req.get('host')
            }
        });
    } catch (error) {
        console.error('❌ Generate session error:', error);
        res.status(500).json({ 
            success: false, 
            message: 'Database error: ' + error.message 
        });
    }
});

// Customer connection
app.post('/api/customer/connect', async (req, res) => {
    try {
        const { name, supportCode, sessionId } = req.body;
        const targetSessionId = sessionId || supportCode;
        
        console.log(`🔄 Customer "${name}" trying to connect to session "${targetSessionId}"`);
        console.log(`🔒 Customer accessing via: ${req.protocol}://${req.get('host')}`);
        
        // Warn if customer is accessing via HTTP
        if (req.protocol === 'http' && req.hostname !== 'localhost' && req.hostname !== '127.0.0.1') {
            console.warn(`⚠️ Customer accessing via HTTP - WebRTC screen sharing will not work!`);
        }
        
        // Check if session exists
        const [sessions] = await pool.execute(
            'SELECT * FROM sessions WHERE id = ? AND status != ?',
            [targetSessionId, 'ended']
        );
        
        if (sessions.length === 0) {
            console.log(`❌ Session "${targetSessionId}" not found or ended`);
            return res.status(404).json({ 
                success: false, 
                message: 'Session not found. Please check your support code.' 
            });
        }
        
        const session = sessions[0];
        console.log(`✅ Found session: ${session.id} (status: ${session.status})`);
        
        // Insert customer
        const [result] = await pool.execute(
            'INSERT INTO customers (session_id, name, support_code, status) VALUES (?, ?, ?, ?)',
            [session.id, name, supportCode, 'connected']
        );
        
        const customerId = result.insertId;
        
        // Update session status to active
        await pool.execute(
            'UPDATE sessions SET status = ? WHERE id = ?',
            ['active', session.id]
        );
        
        // Add system message with protocol warning if needed
        let welcomeMessage = `${name} has joined the support session`;
        if (req.protocol === 'http' && req.hostname !== 'localhost') {
            welcomeMessage += ' (Note: Screen sharing requires HTTPS)';
        }
        
        await pool.execute(
            'INSERT INTO chat_messages (session_id, sender_type, sender_name, message) VALUES (?, ?, ?, ?)',
            [session.id, 'system', 'System', welcomeMessage]
        );
        
        console.log(`✅ Customer "${name}" connected with ID ${customerId}`);
        
        res.json({
            success: true,
            customerId,
            sessionId: session.id,
            message: 'Connected to support successfully',
            protocol: req.protocol,
            canShareScreen: req.protocol === 'https' || req.hostname === 'localhost'
        });
        
        // Notify admin via websocket
        io.to(session.id).emit('customerConnected', {
            sessionId: session.id,
            customer: { 
                id: customerId, 
                name, 
                supportCode, 
                status: 'connected',
                protocol: req.protocol,
                canShareScreen: req.protocol === 'https' || req.hostname === 'localhost'
            }
        });
        
        console.log(`📡 Notified admin about customer connection`);
        
    } catch (error) {
        console.error('❌ Customer connect error:', error);
        res.status(500).json({ 
            success: false, 
            message: 'Database error: ' + error.message 
        });
    }
});

// Get session status
app.get('/api/session/:sessionId/status', async (req, res) => {
    try {
        const { sessionId } = req.params;
        
        console.log(`🔍 Checking status for session: ${sessionId}`);
        
        const [sessions] = await pool.execute(
            'SELECT * FROM sessions WHERE id = ?',
            [sessionId]
        );
        
        if (sessions.length === 0) {
            return res.status(404).json({ success: false, message: 'Session not found' });
        }
        
        const [customers] = await pool.execute(
            'SELECT * FROM customers WHERE session_id = ? AND status = ? ORDER BY connected_at DESC',
            [sessionId, 'connected']
        );
        
        const [screenSharing] = await pool.execute(
            'SELECT * FROM screen_sharing WHERE session_id = ? ORDER BY updated_at DESC LIMIT 1',
            [sessionId]
        );
        
        console.log(`✅ Session ${sessionId}: ${customers.length} connected customers`);
        
        res.json({
            success: true,
            session: sessions[0],
            customers,
            screenSharing: screenSharing[0] || null
        });
    } catch (error) {
        console.error('❌ Get session status error:', error);
        res.status(500).json({ success: false, message: 'Database error: ' + error.message });
    }
});

// Start screen sharing
app.post('/api/screen-share/start', async (req, res) => {
    try {
        const { sessionId, customerId } = req.body;
        
        console.log(`🖥️ Starting screen share for customer ${customerId} in session ${sessionId}`);
        console.log(`🔒 Screen share request via: ${req.protocol}://${req.get('host')}`);
        
        await pool.execute(
            'UPDATE customers SET is_sharing_screen = ? WHERE id = ? AND session_id = ?',
            [true, customerId, sessionId]
        );
        
        await pool.execute(
            'INSERT INTO screen_sharing (session_id, customer_id, status) VALUES (?, ?, ?) ON DUPLICATE KEY UPDATE status = ?, updated_at = CURRENT_TIMESTAMP',
            [sessionId, customerId, 'active', 'active']
        );
        
        console.log(`✅ Screen sharing started for customer ${customerId}`);
        
        res.json({ success: true, message: 'Screen sharing started' });
        
        // Notify admin in the specific session room
        io.to(sessionId).emit('screenShareStarted', { sessionId, customerId });
        
    } catch (error) {
        console.error('❌ Start screen share error:', error);
        res.status(500).json({ success: false, message: 'Database error: ' + error.message });
    }
});

// Stop screen sharing
app.post('/api/screen-share/stop', async (req, res) => {
    try {
        const { sessionId, customerId } = req.body;
        
        console.log(`🛑 Stopping screen share for customer ${customerId} in session ${sessionId}`);
        
        await pool.execute(
            'UPDATE customers SET is_sharing_screen = ? WHERE id = ? AND session_id = ?',
            [false, customerId, sessionId]
        );
        
        await pool.execute(
            'UPDATE screen_sharing SET status = ?, updated_at = CURRENT_TIMESTAMP WHERE session_id = ? AND customer_id = ?',
            ['stopped', sessionId, customerId]
        );
        
        console.log(`✅ Screen sharing stopped for customer ${customerId}`);
        
        res.json({ success: true, message: 'Screen sharing stopped' });
        
        // Notify admin in the specific session room
        io.to(sessionId).emit('screenShareStopped', { sessionId, customerId });
        
    } catch (error) {
        console.error('❌ Stop screen share error:', error);
        res.status(500).json({ success: false, message: 'Database error: ' + error.message });
    }
});

// Request screen sharing
app.post('/api/screen-share/request', async (req, res) => {
    try {
        const { sessionId, adminUsername } = req.body;
        
        console.log(`📱 Admin ${adminUsername} requesting screen share for session ${sessionId}`);
        
        const [customers] = await pool.execute(
            'SELECT * FROM customers WHERE session_id = ? AND status = ? LIMIT 1',
            [sessionId, 'connected']
        );
        
        if (customers.length === 0) {
            return res.status(404).json({ success: false, message: 'No connected customers' });
        }
        
        const customer = customers[0];
        
        await pool.execute(
            'INSERT INTO screen_sharing (session_id, customer_id, status, requested_by) VALUES (?, ?, ?, ?) ON DUPLICATE KEY UPDATE status = ?, requested_by = ?, updated_at = CURRENT_TIMESTAMP',
            [sessionId, customer.id, 'requested', adminUsername, 'requested', adminUsername]
        );
        
        console.log(`✅ Screen share request sent to customer ${customer.name}`);
        
        res.json({ success: true, message: 'Screen share request sent' });
        
        // Notify customer in the specific session room
        io.to(sessionId).emit('screenShareRequested', { sessionId, customerId: customer.id });
        
    } catch (error) {
        console.error('❌ Request screen share error:', error);
        res.status(500).json({ success: false, message: 'Database error: ' + error.message });
    }
});

// Chat messages
app.post('/api/chat/send', async (req, res) => {
    try {
        const { sessionId, senderType, senderName, message } = req.body;
        
        console.log(`💬 Chat message from ${senderName} (${senderType}) in session ${sessionId}`);
        
        await pool.execute(
            'INSERT INTO chat_messages (session_id, sender_type, sender_name, message) VALUES (?, ?, ?, ?)',
            [sessionId, senderType, senderName, message]
        );
        
        res.json({ success: true, message: 'Message sent' });
        
        // Broadcast to all users in the session room
        io.to(sessionId).emit('newChatMessage', {
            sessionId,
            senderType,
            senderName,
            message,
            timestamp: new Date()
        });
        
    } catch (error) {
        console.error('❌ Send chat message error:', error);
        res.status(500).json({ success: false, message: 'Database error: ' + error.message });
    }
});

// Get chat messages
app.get('/api/chat/:sessionId', async (req, res) => {
    try {
        const { sessionId } = req.params;
        
        const [messages] = await pool.execute(
            'SELECT * FROM chat_messages WHERE session_id = ? ORDER BY created_at ASC',
            [sessionId]
        );
        
        console.log(`💬 Retrieved ${messages.length} chat messages for session ${sessionId}`);
        
        res.json({ success: true, messages });
    } catch (error) {
        console.error('❌ Get chat messages error:', error);
        res.status(500).json({ success: false, message: 'Database error: ' + error.message });
    }
});

// Customer heartbeat
app.post('/api/customer/heartbeat', async (req, res) => {
    try {
        const { customerId, sessionId } = req.body;
        
        await pool.execute(
            'UPDATE customers SET last_seen = CURRENT_TIMESTAMP WHERE id = ? AND session_id = ?',
            [customerId, sessionId]
        );
        
        res.json({ success: true });
    } catch (error) {
        console.error('❌ Heartbeat error:', error);
        res.status(500).json({ success: false, message: 'Database error: ' + error.message });
    }
});

// Disconnect customer
app.post('/api/customer/disconnect', async (req, res) => {
    try {
        const { customerId, sessionId } = req.body;
        
        console.log(`👋 Customer ${customerId} disconnecting from session ${sessionId}`);
        
        await pool.execute(
            'UPDATE customers SET status = ? WHERE id = ? AND session_id = ?',
            ['disconnected', customerId, sessionId]
        );
        
        res.json({ success: true, message: 'Customer disconnected' });
        
        // Notify admin in the specific session room
        io.to(sessionId).emit('customerDisconnected', { sessionId, customerId });
        
    } catch (error) {
        console.error('❌ Disconnect customer error:', error);
        res.status(500).json({ success: false, message: 'Database error: ' + error.message });
    }
});

// WebSocket connections with WebRTC signaling support
io.on('connection', (socket) => {
    console.log('🔗 Client connected:', socket.id);
    
    socket.on('joinSession', (sessionId) => {
        socket.join(sessionId);
        console.log(`📡 Socket ${socket.id} joined session ${sessionId}`);
    });

    // WebRTC Signaling Events
    socket.on('webrtc-offer', (data) => {
        console.log(`📡 WebRTC offer for session ${data.sessionId}`);
        // Forward offer to all other clients in the session (mainly customer)
        socket.to(data.sessionId).emit('webrtc-offer', data);
    });

    socket.on('webrtc-answer', (data) => {
        console.log(`📡 WebRTC answer for session ${data.sessionId}`);
        // Forward answer to all other clients in the session (mainly admin)
        socket.to(data.sessionId).emit('webrtc-answer', data);
    });

    socket.on('webrtc-ice-candidate', (data) => {
        console.log(`📡 WebRTC ICE candidate for session ${data.sessionId}`);
        // Forward ICE candidate to all other clients in the session
        socket.to(data.sessionId).emit('webrtc-ice-candidate', data);
    });
    
    socket.on('disconnect', () => {
        console.log('🔌 Client disconnected:', socket.id);
    });
});

// Cleanup inactive customers
setInterval(async () => {
    try {
        const [result] = await pool.execute(
            'UPDATE customers SET status = ? WHERE status = ? AND last_seen < DATE_SUB(NOW(), INTERVAL 2 MINUTE)',
            ['disconnected', 'connected']
        );
        
        if (result.affectedRows > 0) {
            console.log(`🧹 Cleaned up ${result.affectedRows} inactive customers`);
        }
    } catch (error) {
        console.error('❌ Cleanup error:', error);
    }
}, 5 * 60 * 1000);

// Serve the frontend
app.get('/', (req, res) => {
    res.sendFile(path.join(__dirname, 'public', 'index.html'));
});

// Health check endpoint with protocol information
app.get('/health', async (req, res) => {
    const isDbConnected = await testDatabaseConnection();
    
    let tableStatus = {};
    try {
        const connection = await pool.getConnection();
        const [tables] = await connection.execute('SHOW TABLES');
        const tableNames = tables.map(t => Object.values(t)[0]);
        
        const requiredTables = ['sessions', 'customers', 'chat_messages', 'screen_sharing'];
        requiredTables.forEach(table => {
            tableStatus[table] = tableNames.includes(table) ? 'exists' : 'missing';
        });
        
        connection.release();
    } catch (error) {
        tableStatus = { error: error.message };
    }
    
    res.json({
        status: isDbConnected ? 'healthy' : 'unhealthy',
        database: isDbConnected ? 'connected' : 'disconnected',
        databaseName: 'meeting',
        tables: tableStatus,
        webrtc: 'signaling-enabled',
        protocol: {
            request: req.protocol,
            secure: getSecureProtocol(req),
            headers: {
                'x-forwarded-proto': req.get('x-forwarded-proto'),
                'x-forwarded-ssl': req.get('x-forwarded-ssl'),
                'host': req.get('host')
            }
        },
        environment: process.env.NODE_ENV || 'development',
        timestamp: new Date().toISOString()
    });
});

// Start server
const PORT = process.env.PORT || 3003;

async function startServer() {
    console.log('🚀 Starting Remote Support Server with WebRTC...');
    console.log('📊 Using "meeting" database');
    console.log('🔒 HTTPS URL generation: ENABLED');
    console.log('🌍 Environment:', process.env.NODE_ENV || 'development');
    
    // Test database connection first
    const isConnected = await testDatabaseConnection();
    if (!isConnected) {
        console.error('❌ Cannot start server - database connection failed');
        console.error('💡 Make sure MySQL is running and credentials are correct');
        process.exit(1);
    }
    
    try {
        // Initialize database
        await initializeDatabase();
        
        // Start server
        server.listen(PORT, () => {
            console.log('');
            console.log('🎉 Remote Support Server with WebRTC is running!');
            console.log(`🌐 Frontend: http://localhost:${PORT}`);
            console.log(`🔒 HTTPS Frontend: https://localhost:${PORT} (if SSL configured)`);
            console.log(`🏥 Health Check: http://localhost:${PORT}/health`);
            console.log('📊 Database: MySQL (meeting)');
            console.log('🔧 WebSocket: Socket.IO with WebRTC signaling');
            console.log('📹 Video: Real-time screen sharing via WebRTC');
            console.log('🔒 URL Generation: HTTPS-first for WebRTC compatibility');
            console.log('');
            console.log('✅ Ready to accept connections! 🚀');
            console.log('');
            console.log('📋 IMPORTANT - HTTPS Setup:');
            console.log('   🔒 For WebRTC screen sharing to work:');
            console.log('   1. Configure SSL certificate for your domain');
            console.log('   2. Ensure nginx/apache forwards HTTPS properly');
            console.log('   3. Share URLs will now be HTTPS automatically');
            console.log('   4. Test at: https://meeting.creatorschoice.in');
            console.log('');
            console.log('🎯 Next steps:');
            console.log('   1. Configure SSL certificate');
            console.log('   2. Access via https://meeting.creatorschoice.in');
            console.log('   3. Generate session - URLs will now be HTTPS');
            console.log('   4. Share HTTPS URL with customers');
        });
    } catch (error) {
        console.error('❌ Failed to initialize database:', error.message);
        process.exit(1);
    }
}

// Start the server
startServer().catch(error => {
    console.error('❌ Failed to start server:', error);
    process.exit(1);
});